/**************************************************************************
FILE          : qnx-interrupt.c
PROGRAMMER    : Peter C. Chapin
LAST REVISION : 2008-11-02

This file contains a skeleton program to illustrate one way for dealing
with interrupts in a QNX system. This code is pretty much the sample
provided with the QNX documenation.

Please send comments or bug reports to

     Peter C. Chapin
     Computer Information Systems Department
     Vermont Technical College
     Randolph Center, VT 05061
     Peter.Chapin@vtc.vsc.edu
**************************************************************************/

#include <pthread.h>
#include <sys/siginfo.h>
#include <sys/neutrino.h>

#define IRQ4 4

// Declaration of ISR function.
const struct sigevent *isr(void *area, int id);

struct sigevent event;


// This is the ISR. If the work you need to do here is trivial, you
// might consider using InterruptAttachEvent() instead of Interrupt-
// Attach() and elminiate this ISR completely. Consult the QNX
// documentation for more information.
//
const struct sigevent *isr(void *arg, int id)
{
  // Look at the hardware to see if it caused the interrupt. If not,
  // simply return NULL.

  // In a level-sensitive environment, clear the cause of the interrupt,
  // or alternatively issue InterruptMask() to disable the PIC from
  // reinterrupting the kernel.
  
  // If you need to access a data structure that is shared with a
  // "normal" thread use InterruptLock and InterruptUnlock around the
  // critical section.

  // Return a pointer to an event structure (initialized by main) that
  // contains SIGEV_INTR as its notification type. This causes the
  // InterruptWait in "hardware_thread" to unblock.
  //
  return &event;
}


// This thread interacts with the hardware. Most of the interrupt handling
// is done here (do as little as possible in the actual interrupt service
// routine above).
//
void *hardware_thread(void *arg)
{
  int interrupt_id;

  // Used for modifying the scheduling policy for this thread.
  int policy;
  struct sched_param parameters;

    
  // Enable I/O privilege.
  ThreadCtl(_NTO_TCTL_IO, NULL);

  // Initialize the hardware as necessary.

  // Attach the ISR to IRQ 4 (for example).
  interrupt_id = InterruptAttach(IRQ4, isr, NULL, 0, 0);
  if (interrupt_id == -1) return NULL;

  // Do we want to boost this thread's priority? Probably. However this is
  // not advised during development since it can cause system hangs if
  // a high priority thread runs wild.
  //
  // pthread_getschedparam(pthread_self(), &policy, &parameters);
  // policy = SCHED_FIFO;
  // parameters.sched_priority += 5;
  // pthread_setschedparam(pthread_self(), policy, &parameters);
  

  // Now service the hardware when the ISR says to do so.
  while (1) {
    InterruptWait(0, NULL);
    // At this point, when InterruptWait() unblocks, the ISR has
    // returned a SIGEV_INTR indicating that some form of work needs to
    // be done.

    // Do the work (manipulate the hardware). If this thread shares any
    // data with the ISR it should call InterruptLock() and Interrupt-
    // Unlock() around the critical section.

    // If the ISR did an InterruptMask(), then this thread should clear
    // the source of the interrupt and then do an InterruptUnmask() to
    // allow interrupts from the hardware.
  }

  // Detach the ISR from the interrupt when done.
  InterruptDetach(interrupt_id);
}


// ============
// Main Program
// ============

int main(void)
{
  pthread_t threadID;
  
  // Perform initializations, etc.
  SIGEV_INTR_INIT(&event);

  // Start up a thread that is dedicated to interrupt processing.
  pthread_create(&threadID, NULL, hardware_thread, NULL);

  // Perform other processing, as appropriate.
  
  pthread_join(threadID, NULL);
  return 0;
}
